/*
 * COpenGL.cpp
 *
 *  Created on: 04.06.2009
 *      Author: gerstrong
 */
#include "../Configurator.h"

#ifdef USE_OPENGL

#include "COpenGL.h"
#include <base/video/CVideoDriver.h>
#include <base/CInput.h> // for CInput::renderOverlay
#include "graphics/GsGraphics.h"
#include <base/GsLogging.h>
#include <base/GsApp.h>


// gamerect is the base resolution for the game which is scaled with the filter
// depending on what resolution has been chosen, it is mostly 320x200 or 320x240
COpenGL::COpenGL(const CVidConfig &VidConfig) :
CVideoEngine(VidConfig),
m_texparam(GL_TEXTURE_2D),
    m_GameScaleDim(m_VidConfig.mGameRect.dim.x*sFiltToNum(m_VidConfig.m_ScaleXFilter),
                   m_VidConfig.mGameRect.dim.y*sFiltToNum(m_VidConfig.m_ScaleXFilter)),
m_GamePOTScaleDim(getPowerOfTwo(m_GameScaleDim.w), getPowerOfTwo(m_GameScaleDim.h))
{}

void COpenGL::setUpViewPort(const GsRect<Uint16> &viewRect)
{
    const auto displayRect = m_VidConfig.mDisplayRect;
    glViewport(displayRect.pos.x, displayRect.pos.y,
               displayRect.dim.x, displayRect.dim.y);

    const auto width = static_cast<GLsizei>(viewRect.dim.x);
    const auto height = static_cast<GLsizei>(viewRect.dim.y);
    const auto xpos = static_cast<GLint>(viewRect.pos.x);
    const auto ypos = static_cast<GLint>(viewRect.pos.y);
    glOrtho(xpos, ypos, width, height, -1, 1);
}

void COpenGL::resizeDisplayScreen(const GsRect<Uint16>& newRect)
{
    auto &log = gLogging;

    try
    {
        const auto &asp = m_VidConfig.mAspectCorrection.dim;
        updateActiveArea(newRect, asp);

        const auto &viewport = m_VidConfig.mDisplayRect.dim;

        mActiveAreaRect.pos.x = (viewport.x-newRect.dim.x)/2;
        mActiveAreaRect.pos.y = (viewport.y-newRect.dim.y)/2;

        setUpViewPort(mActiveAreaRect);
    }
    catch (...)
    {
        log << "unhandled exception." << CLogFile::endl;
    }
}

void COpenGL::collectSurfaces()
{

}

void COpenGL::clearSurfaces()
{    
    mpScreenSfc->fillRGB(mClearColor.r, mClearColor.b, mClearColor.g);
}


static void createTexture(GLuint& tex, GLint oglfilter, GLsizei potwidth, GLsizei potheight, bool withAlpha = false)
{
	glGenTextures(1, &tex);
	glBindTexture(GL_TEXTURE_2D, tex);
	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, oglfilter);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, oglfilter);

	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

	if(withAlpha)
        glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, potwidth, potheight, 0, GL_RGBA, GL_UNSIGNED_BYTE, nullptr);
	else
        glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, potwidth, potheight, 0, GL_RGB, GL_UNSIGNED_BYTE, nullptr);
}

bool COpenGL::init()
{    
    if(!CVideoEngine::init())
        return false;

    auto &log = gLogging;

    // Set by user the chosen filter
    GLint oglfilter = GL_LINEAR;
    if(m_VidConfig.mRenderScQuality == CVidConfig::RenderQuality::NEAREST)
    {
        oglfilter = GL_NEAREST;
    }

    Uint32 flags = SDL_WINDOW_OPENGL;

    if(m_VidConfig.mFullscreen)
        flags |= SDL_WINDOW_FULLSCREEN_DESKTOP;
    else
        flags |= (SDL_WINDOW_SHOWN | SDL_WINDOW_RESIZABLE);


    // set the opengl context version
    SDL_GL_SetAttribute(SDL_GL_CONTEXT_MAJOR_VERSION, 2);
    SDL_GL_SetAttribute(SDL_GL_CONTEXT_MINOR_VERSION, 1);

    // turn on double buffering set the depth buffer to 24 bits
    // you may need to change this to 16 or 32 for your system
    SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);
    SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE, 24);


    window = SDL_CreateWindow(gApp.getName().c_str(),
                              SDL_WINDOWPOS_CENTERED,
                              SDL_WINDOWPOS_CENTERED,
                              m_VidConfig.mDisplayRect.dim.x,
                              m_VidConfig.mDisplayRect.dim.y,
                              flags);


    glcontext = SDL_GL_CreateContext(window);

    // Sync buffer swap with monitor's vertical refresh rate
    if(m_VidConfig.mVSync)
    {
        if( SDL_GL_SetSwapInterval( 1 ) < 0 )
        {
            log << "Warning: Unable to set VSync! SDL Error: "
                     << SDL_GetError()
                     << CLogFile::endl;
        }
    }
    
	// Set clear colour
	glClearColor(0,0,0,0);
	
	// Set projection
	glMatrixMode( GL_PROJECTION );
	glLoadIdentity();

    auto error = glGetError();
    if( error != GL_NO_ERROR )
    {
        log << "OpenGL Init(): "<< error << CLogFile::endl;
        return false;
    }
    
#if TARGET_OS_IPHONE || TARGET_IPHONE_SIMULATOR	// TODO: dont check for iphone but for opengles
#define glOrtho glOrthof
#endif
	glOrtho( 0.0f, 1.0f, 1.0f, 0.0f, 0.0f, 1.0f );
    
	// Now Initialize modelview matrix
	glMatrixMode( GL_MODELVIEW );
	glLoadIdentity();

    error = glGetError();
    if( error != GL_NO_ERROR )
    {
        log << "OpenGL Init(): "<< error << CLogFile::endl;
        return false;
    }

    // Setup the view port for the first time
#if TARGET_OS_IPHONE || TARGET_IPHONE_SIMULATOR
    m_VidConfig.mDisplayRect.pos= {0,0};
    m_VidConfig.mDisplayRect.pos= {480,320};
#else
#endif
    resizeDisplayScreen(m_VidConfig.mDisplayRect);

    /*Using the standard OpenGL API for specifying a 2D texture
     image: glTexImage2D, glSubTexImage2D, glCopyTexImage2D,
     and glCopySubTexImage2D.  The target for these commands is
     GL_TEXTURE_RECTANGLE_ARB though.
     
     This is similar to how the texture cube map functionality uses the 2D
     texture image specification API though with its own texture target.
     
     The texture target GL_TEXTURE_RECTANGLE_ARB should also
     be used for glGetTexImage, glGetTexLevelParameteriv, and
     glGetTexLevelParameterfv.*/
    
	// Enable Texture loading for the blit screen
	glEnable(m_texparam);

    createTexture(m_texture, oglfilter, m_GamePOTScaleDim.w, m_GamePOTScaleDim.h);

    if(sFiltToNum(m_VidConfig.m_ScaleXFilter) <= 1)
	{	// In that case we can do a texture based rendering
        createTexture(m_texFX, oglfilter, m_GamePOTScaleDim.w, m_GamePOTScaleDim.h, true);
	}
	
	// If there were any errors
    error = glGetError();
	if( error != GL_NO_ERROR)
	{
        log << "OpenGL Init(): "<< error << CLogFile::endl;
        SDL_DestroyWindow(window);
        SDL_GL_DeleteContext(glcontext);
		return false;
	}

    log << "OpenGL Init(): Interface successfully opened!" << CLogFile::endl;

    GLfloat light_ambient[] = { 1.0, 1.0, 1.0, 1.0 };
    GLfloat light_diffuse[] = { 1.0, 1.0, 1.0, 1.0 };
    GLfloat light_specular[] = { 1.0, 1.0, 1.0, 1.0 };
    GLfloat light_position[] = { 0.0, 0.0, 1.0, 0.0 };

    glLightfv(GL_LIGHT0, GL_AMBIENT, light_ambient);
    glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse);
    glLightfv(GL_LIGHT0, GL_SPECULAR, light_specular);
    glLightfv(GL_LIGHT0, GL_POSITION, light_position);

    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
	
	return true;
}

static void renderTexture([[maybe_unused]]GLuint texture, bool withAlpha = false)
{
	if(withAlpha)
	{
		glColor4f(1.0f, 1.0f, 1.0f, 0.8f);
        //glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_ALPHA);
        glBlendFunc(GL_ONE, GL_ONE);
	}
	else
	{
		glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
		glBlendFunc(GL_ONE, GL_ZERO);
	}
	
	//Finally draw the arrays of the surface.
	glDrawArrays(GL_TRIANGLE_FAN, 0, 4);
}

void COpenGL::loadSurface(GLuint texture, SDL_Surface* surface)
{
	glBindTexture (m_texparam, texture);
	GLint internalFormat, externalFormat;

#if !TARGET_OS_IPHONE && !TARGET_IPHONE_SIMULATOR // iPhone always used 32 bits; also GL_BGR is not defined
	if(surface->format->BitsPerPixel == 24)
	{
		internalFormat = GL_RGB;
		externalFormat = GL_BGR;
	}
	else
#endif
	{	// we assume RGBA
		internalFormat = GL_RGBA;
		externalFormat = GL_BGRA;
	}

    mpScreenSfc->lock();

	glTexImage2D(m_texparam, 0, internalFormat,
                mpScreenSfc->width(),
                mpScreenSfc->height(),
				0, externalFormat,
                GL_UNSIGNED_BYTE, mpScreenSfc->getSDLSurface()->pixels);

    mpScreenSfc->unlock();
}

void COpenGL::transformScreenToDisplay()
{
	glEnable(GL_TEXTURE_2D);
	// Set up an array of values to use as the sprite vertices.
	GLfloat vertices[] =
	{
		0, 0,
		1, 0,
		1, 1,
		0, 1,
	};

	// Set up an array of values for the texture coordinates.
	GLfloat texcoords[] =
	{
		0, 0,
		1, 0,
		1, 1,
		0, 1,
	};

	//Render the vertices by pointing to the arrays.
    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);

	glVertexPointer(2, GL_FLOAT, 0, vertices);
	glTexCoordPointer(2, GL_FLOAT, 0, texcoords);

	glEnable(GL_BLEND);

    loadSurface(m_texture, mpScreenSfc->getSDLSurface());
	renderTexture(m_texture);

	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	glDisable(GL_BLEND);
	glDisable(GL_TEXTURE_2D);

	gInput.renderOverlay();

#if SDL_VERSION_ATLEAST(2, 0, 0)
    SDL_GL_SwapWindow(window);
#else
    SDL_GL_SwapBuffers();
#endif

    clearSurfaces();
}

void COpenGL::setLightIntensity(const float intensity)
{
    float r,g,b;

    glDisable(GL_LIGHT0);

    // TODO: This a temporary solution for just now.
    // We should pass the different color channels as parameters in future
    r = intensity;
    g = intensity;
    b = intensity;

    GLfloat light_ambient[] = { r, g, b, 1.0 };
    GLfloat light_diffuse[] = { r, g, b, 1.0 };
    GLfloat light_specular[] = { r, g, b, 1.0 };
    GLfloat light_position[] = { 0.0, 0.0, 1.0, 0.0 };

    glLightfv(GL_LIGHT0, GL_AMBIENT, light_ambient);
    glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse);
    glLightfv(GL_LIGHT0, GL_SPECULAR, light_specular);
    glLightfv(GL_LIGHT0, GL_POSITION, light_position);

    glEnable(GL_LIGHT0);
}

#endif // USE_OPENGL
